/*
 *  ======== TMP117.h ========
 *  TMP117 C Interface
 */
#ifndef ti_sensors_TMP117__include
#define ti_sensors_TMP117__include 1

#include <stdint.h>

/* support C++ sources */
#ifdef __cplusplus
extern "C" {
#endif

#define TMP117_TEMP 0x00U
#define TMP117_CONFIG 0x01U
#define TMP117_CONFIG_ALERT_HIGH_ENABLE 0x8000U
#define TMP117_CONFIG_ALERT_HIGH_DISABLE 0x0000U
#define TMP117_CONFIG_ALERT_LOW_ENABLE 0x4000U
#define TMP117_CONFIG_ALERT_LOW_DISABLE 0x0000U
#define TMP117_CONFIG_CONV_RDY_ENABLE 0x2000U
#define TMP117_CONFIG_CONV_RDY_DISABLE 0x0000U
#define TMP117_CONFIG_EEPROM_BUSY_ENABLE 0x1000U
#define TMP117_CONFIG_EEPROM_BUSY_DISABLE 0x0000U
#define TMP117_CONFIG_CONV_MODE_CC 0x0000U
#define TMP117_CONFIG_CONV_MODE_SD 0x0400U
#define TMP117_CONFIG_CONV_CYCLE_15p5ms 0x0000U
#define TMP117_CONFIG_CONV_CYCLE_125ms 0x0080U
#define TMP117_CONFIG_CONV_CYCLE_250ms 0x0100U
#define TMP117_CONFIG_CONV_CYCLE_500ms 0x0180U
#define TMP117_CONFIG_CONV_CYCLE_1s 0x0200U
#define TMP117_CONFIG_CONV_CYCLE_4s 0x0280U
#define TMP117_CONFIG_CONV_CYCLE_8s 0x0300U
#define TMP117_CONFIG_CONV_CYCLE_16s 0x0380U
#define TMP117_CONFIG_AVG_MODE_NONE 0x0000U
#define TMP117_CONFIG_AVG_MODE_8AVG 0x0020U
#define TMP117_CONFIG_AVG_MODE_32AVG 0x0040U
#define TMP117_CONFIG_AVG_MODE_64AVG 0x0060U
#define TMP117_CONFIG_TnA_ALERT 0x0000U
#define TMP117_CONFIG_TnA_THERM 0x0010U
#define TMP117_CONFIG_ALERT_POL_LOW 0x0000U
#define TMP117_CONFIG_ALERT_POL_HIGH 0x0008U
#define TMP117_CONFIG_CRDY_EN_ALERT 0x0000U
#define TMP117_CONFIG_CRDY_EN_DRDY 0x0004U
#define TMP117_CONFIG_SOFT_RESET_ENABLE 0x0002U
#define TMP117_CONFIG_SOFT_RESET_DISABLE 0x0000U
#define TMP117_CONFIG_Reserved_ENABLE 0x0001U
#define TMP117_CONFIG_Reserved_DISABLE 0x0000U
#define TMP117_THIGH 0x02U
#define TMP117_TLOW 0x03U
#define TMP117_EEPROM 0x04U
#define TMP117_EEPROM_EUN_ENABLE 0x8000U
#define TMP117_EEPROM_EUN_DISABLE 0x0000U
#define TMP117_EEPROM_EEPROM_BUSY_ENABLE 0x4000U
#define TMP117_EEPROM_EEPROM_BUSY_DISABLE 0x0000U
#define TMP117_EEPROM1 0x05U
#define TMP117_EEPROM2 0x06U
#define TMP117_EEPROM3 0x08U
#define TMP117_OFFSET_TEMP 0x07U
#define TMP117_ID 0x0FU

#define TMP117_OS_ENABLE 0x0C00U

/*
 *  ======== TMP117_State ========
 *  Initial configuration state for a TMP117 sensor
 */
typedef struct TMP117_State {
    uint16_t config;       /* CONFIG register */

    uint16_t thigh;        /* THIGH and TLOW limit registers */
    uint16_t tlow;

    uint16_t offsetTemp;   /* OFFSET_TEMP register */

    uint16_t eeprom1;      /* EEPROMx registers */
    uint16_t eeprom2;
    uint16_t eeprom3;

    uint8_t busId;         /* I2C bus id */
    uint8_t devAddr;       /* TMP117 sensor address on the I2C bus */

    uint8_t nvmCommit;     /* 1 => save settings to EEPROM, 0 => don't */
    uint8_t preserveNIST;  /* 1 => don't overwrite NIST date, 0 => allow overwrite */

    uint32_t osWait;       /* One shot conversion time (in ms) */

} TMP117_State;

/*
 *  ======== TMP117_Handle ========
 *  First argument to all TMP117 methods
 */
typedef TMP117_State *TMP117_Handle;

/*
 *  ======== TMP117_config ========
 *  Configure device with current settings
 */
extern void TMP117_config(TMP117_Handle sensor);

/*
 *  ======== TMP117_read ========
 *  Read raw temperature register
 *
 *  Returns a signed result in units of 0.0078125 degrees C.  For example,
 *  a return value of 128 represents 1 degree Celsius.
 *
 *  Conversion functions are provided to convert a raw value returned by this
 *  function to value in "natural" units.
 *
 *  For example, TMP117_toMilliCelsius(value) converts a raw
 *  temperature value into an integral temperature value in units of
 *  milli-degrees Celsius.
 *  
 *  @param sensor  handle to a TMP117 state object
 *  @returns       a 32-bit signed result in units of 0.0078125 degrees C
 *
 *  @see TMP117_toIntCelsius TMP117_toFloatCelsius
 *       TMP117_toMilliCelsius
 */
extern int32_t TMP117_read(TMP117_Handle sensor);

/*
 *  ======== TMP117_toIntCelsius ========
 *  Convert raw temperature register value to whole degrees Celsius
 *
 *  This function rounds values to the nearest integer.
 *
 *  @param raw    32-bit sign extended temperature value read from the
 *                sensor's read function: TMP117_read()
 *  @returns      signed 32-bit integer representation of temperature in 
 *                degrees Celsius, rounded to the nearest whole number
 *  @see          TMP117_read
 */
extern int32_t TMP117_toIntCelsius(int32_t raw);

/*
 *  ======== TMP117_toMilliCelsius ========
 *  Convert raw field temperature to whole milli-degrees Celsius
 *
 *  This function truncates rather than rounds values.
 *
 *  @param raw    32-bit sign extended temperature value read from the
 *                sensor's read function: TMP117_read()
 *  @returns      signed 32-bit integer representation of temperature in 
 *                milli-degrees Celsius
 *  @see          TMP117_read
 */
extern int32_t TMP117_toMilliCelsius(int32_t raw);

/*
 *  ======== TMP117_toFloatCelsius ========
 *  Convert raw temperature value to fractional degrees Celsius
 *
 *  @param raw    32-bit sign extended temperature value read from the
 *                sensor's read function: TMP117_read()
 *  @returns      floating point representation of temperature in degrees
 *                Celsius
 *  @see          TMP117_read
 */
extern float TMP117_toFloatCelsius(int32_t raw);


/* support C++ sources */
#ifdef __cplusplus
}
#endif

#endif

